#!/bin/bash

if [ "$(id -u)" == "0" ]; then
    echo -e "\nGlobalProtect cannot be installed as a root user."
    echo -e "Please install the agent as a user with admin privileges.\n"
    exit 1
fi

# Determine command type
cmd_type="ui"
if [ $# -gt 0 ]; then
    case $1 in
        --cli-only)
            cmd_type="cli-only";;
        --arm)
            cmd_type="arm";;
        --aarch64)
            cmd_type="aarch64";;
        --help)
            ;&
        *)
            cmd_type="usage";;
    esac
fi

# Determine Linux Distro and Version
. /etc/os-release

if [ $ID == "ubuntu" ]; then
    linux_ver=${VERSION_ID:0:2}
elif [[ $ID == "rhel" || $ID == "centos" ]]; then
    linux_ver=${VERSION_ID:0:1}
fi

case $cmd_type in
    cli-only)
        # CLI Only Install
        case $ID in
            ubuntu)
                case $linux_ver in
                    14)
                        ;&
                    16)
                        ;&
                    18)
                        echo "Error: Unsupported Ubuntu version: $linux_ver"
                        exit
                        ;;
                     *)
                        sudo -E apt-get install -y ./GlobalProtect_deb-*.deb;;
                esac
                ;;
            rhel)
                ;&
            fedora)
                ;&
            rocky)
                ;&
            centos)
                # Check if old GP package installed
                yum_output=$(yum list installed | grep globalprotect)
                if [[ $yum_output == *"globalprotect.x86"* ]]; then
                    echo "Older globalprotect detected...uninstalling..."
                    sudo yum -y remove globalprotect
                fi
                
                case $linux_ver in
                    7)
                        echo "Error: Unsupported Linux version: $linux_ver"
                        exit
                        ;;
                    *)
                        sudo -E yum -y install ./GlobalProtect_rpm-*;;
                esac
                ;;
            *)
                echo "Error: Unsupported Linux Distro: $ID"
                exit
                ;;
        esac
        ;;

    arm)
        # ARM Install
        case $ID in
            ubuntu)
                case $linux_ver in
                    14)
                        ;&
                    16)
                        ;&
                    18)
                        echo "Error: Unsupported Ubuntu version: $linux_ver"
                        exit
                        ;;
                     *)
                        sudo -E apt-get install -y ./GlobalProtect_deb_arm*.deb;;
                esac
                ;;
            rhel)
                ;&
            fedora)
                ;&
            rocky)
                ;&
            centos)
                # Check if old GP package installed
                yum_output=$(yum list installed | grep globalprotect)
                if [[ $yum_output == *"globalprotect_arm.x86"* ]]; then
                    echo "Older globalprotect detected...uninstalling..."
                    sudo yum -y remove globalprotect_arm
                fi
                
                case $linux_ver in
                    7)
                        echo "Error: Unsupported Linux version: $linux_ver"
                        exit
                        ;;
                    *)
                        sudo -E yum -y install ./GlobalProtect_rpm_arm*;;
	            esac
	            ;;
            *)
                echo "Error: Unsupported Linux Distro: $ID"
                exit
                ;;
        esac
        ;;
		
    aarch64)
        # AARCH64 Install
        case $ID in
            ubuntu)
                case $linux_ver in
                    14)
                        ;&
                    16)
                        ;&
                    18)
                        echo "Error: Unsupported Ubuntu version: $linux_ver"
                        exit
                        ;;
                     *)
                        sudo -E apt-get install -y ./GlobalProtect_deb_aarch64*.deb;;
                esac
                ;;
            rhel)
                ;&
            fedora)
                ;&
            rocky)
                ;&
            centos)
                # Check if old GP package installed
                yum_output=$(yum list installed | grep globalprotect)
                if [[ $yum_output == *"globalprotect_arm.x86"* ]]; then
                    echo "Older globalprotect detected...uninstalling..."
                    sudo yum -y remove globalprotect_aarch64
                fi
                
                case $linux_ver in
                    7)
                        echo "Error: Unsupported Linux version: $linux_ver"
                        exit
                        ;;
                    *)
                        sudo -E yum -y install ./GlobalProtect_rpm_aarch64*;;
                esac
                ;;
            *)
                echo "Error: Unsupported Linux Distro: $ID"
                exit
                ;;
        esac
        ;;

    ui)
        if [ "$XDG_SESSION_TYPE" = "wayland" ]; then
            echo "Warning: The GlobalProtect UI agent experience will be degraded with Wayland enabled."
            echo "Please switch to X11 for seamless operation."
            echo "Wayland is not currently supported for GP UI installation."
            read -p "Proceed? (y/n): " response

            case $response in
                [yY]*)
                    echo "Continuing..."
                    ;;
                [nN]*)
                    echo "Exiting..."
                    exit 1
                    ;;
                *)
                    echo "Invalid input. Please enter y or n."
                    exit 1
                    ;;
            esac
        fi

        # UI Install
        case $ID in
            ubuntu)
                case $linux_ver in
                    14)
                        ;&
                    16)
                        ;&
                    18)
                        echo "Error: Unsupported Ubuntu version: $linux_ver"
                        exit
                        ;;
                    20)
                        sudo apt-get install -y gnome-tweak-tool gnome-shell-extension-top-icons-plus
                        sudo -E apt-get install -y ./GlobalProtect_UI_deb*.deb
                        gnome-extensions enable TopIcons@phocean.net
                        gsettings set org.gnome.shell.extensions.topicons tray-pos 'right'
                        ;;
                    22)
                        sudo apt-get install -y gnome-shell-extension-manager gnome-shell-extension-appindicator
                        sudo -E apt-get install -y ./GlobalProtect_UI_deb*.deb
                        ;;
                    *)
                        sudo -E apt-get install -y ./GlobalProtect_UI_deb*.deb;;
                esac
                ;;
            rhel)
                ;&
            fedora)
                ;&
            rocky)
                ;&
            centos)
                # Check if old GP package installed
                yum_output=$(yum list installed | grep globalprotect)
                if [[ $yum_output == *"globalprotect_UI.x86"* ]]; then
                    echo "Older globalprotect detected...uninstalling..."
                    sudo yum -y remove globalprotect_UI
                fi
            
                # RHEL Package Dependencies
                if [ "$ID" = "centos" ]; then
                    sudo yum -y install epel-release
                elif [ "$ID" = "rhel" ]; then
                    if [ "$linux_ver" = "7" ]; then
                        sudo yum -y install https://dl.fedoraproject.org/pub/epel/epel-release-latest-7.noarch.rpm
                    elif [ "$linux_ver" = "8" ]; then
                        sudo yum -y install https://dl.fedoraproject.org/pub/epel/epel-release-latest-8.noarch.rpm
                    elif [ "$linux_ver" = "9" ]; then
                        sudo yum -y install https://dl.fedoraproject.org/pub/epel/epel-release-latest-9.noarch.rpm
                    else
                        echo "Error: Unsupported RHEL version: $linux_ver"
                        exit
                    fi
                fi

                sudo yum -y install qt5-qtwebkit wmctrl

                # Gnome Shell Extensions Install
                if [ "$ID" = "rhel" ]; then
                    if [ "$linux_ver" = "8" ]; then
                        sudo yum -y install gnome-shell-extension-topicons-plus gnome-tweaks
                        gnome-shell-extension-tool -e TopIcons@phocean.net
                    elif [ "$linux_ver" = "9" ]; then
                        sudo yum -y install gnome-shell-extension-top-icons
                        gnome-extensions enable top-icons@gnome-shell-extensions.gcampax.github.com
                    fi
                elif [ "$ID" = "fedora" ]; then
                    sudo yum -y install gnome-shell-extension-appindicator gnome-tweaks
                    gnome-extensions enable appindicatorsupport@rgcjonas.gmail.com
                fi

                # Install
                if [ "$ID" = "fedora" ]; then
                    sudo -E rpm -ivh ./GlobalProtect_UI_rpm-*
                else
                    sudo -E yum -y install ./GlobalProtect_UI_rpm-*
                fi
                ;;
            *)
                echo "Error: Unsupported Linux Distro: $ID"
                exit
                ;;
        esac

        # Set GlobalProtect Icon in the Favorites Bar	
        gsettings set org.gnome.shell favorite-apps "$(gsettings get org.gnome.shell favorite-apps | sed s/.$//), 'globalprotect.desktop']"
        echo "GlobalProtect Installation Completed"
        ;;

    usage)
        ;&
    *)
        echo "Usage: $ ./gp_install [--cli-only | --arm | --help]"
        echo "  --cli-only: CLI Only"
        echo "  --arm:      ARM           (32 bit)"
        echo "  --aarch64:  ARM64/aarch64 (64 bit)"
        echo "  default:    UI"
        ;;	
esac
